<?php

/**
 * Get recent klines (candlesticks) from Binance Futures.
 *
 * @param string $symbol Symbol like BTCUSDT
 * @param string $interval Kline interval (e.g., 1m, 5m, 15m, 1h)
 * @param int $limit Number of candles to fetch
 * @return array Kline data
 */
function getBinanceKlines($symbol = "BTCUSDT", $interval = "15m", $limit = 20) {
    $url = "https://fapi.binance.com/fapi/v1/klines?symbol=$symbol&interval=$interval&limit=$limit";

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    $response = curl_exec($ch);
    curl_close($ch);

    return json_decode($response, true) ?? [];
}

/**
 * Calculate dynamic stop-loss for a short position.
 *
 * @param float $entryPrice Short entry price
 * @param array $klines Recent candle data
 * @param float $bufferPercent Extra buffer above highest high
 * @return float|null Stop-loss price
 */
function calculateDynamicShortSL($entryPrice, $klines, $bufferPercent = 0.3) {
    if (empty($klines)) return null;

    // Extract highs from candles
    //$highs = array_map(fn($k) => floatval($k[2]), $klines); // index 2 = high
    $highs = array_map(function($k) {
        return floatval($k[3]); // Closing price
    }, $klines);
    $highestHigh = max($highs);

    // Add buffer (e.g., 0.3%) above resistance
    $stopLoss = $highestHigh + ($highestHigh * $bufferPercent / 100);

    // Make sure SL is above entry (as it's a short)
    return ($stopLoss > $entryPrice) ? round($stopLoss, 5) : round($entryPrice + ($entryPrice * 0.01), 5); // fallback 1% above
}

// === Example Usage ===

if(!empty($_GET['code'])){
    $symbol = $_GET['code'];
}else{
    $symbol = "GUNUSDT";
}
if(!empty($_GET['price'])){
    $entryPrice = $_GET['price'];
}else{
    $entryPrice = 0.0578;
}

$klines = getBinanceKlines($symbol, "1h", 20);
$sl = calculateDynamicShortSL($entryPrice, $klines, 0.3);

if ($sl) {
    echo "Dynamic Stop-Loss for SHORT $symbol: $" . $sl;
} else {
    echo "Failed to calculate SL.";
}


